/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0, 0.0,0.0)

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

#define MAX_STEPS  128 // try to experiment with numb of steps
#define THRESHOLD .01
#define NUM_BALLS  30.

#define OCTAVES  8.0

const float fogDensity = 0.17;

struct objectInfo{
    float hill;
    vec2 id;
    float timeOffset;
    int numBumps;
    float inBump;
};

float rand(vec2 co){
   return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}

float rand2(vec2 co){
   return fract(cos(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}


float object11(vec3 a, vec3 c, float r) {
    return length(a - c) - r;
}

float valueNoiseSimple(vec2 vl) {
   float minStep = 1.0 ;

   vec2 grid = floor(vl);
   vec2 gridPnt1 = grid;
   vec2 gridPnt2 = vec2(grid.x, grid.y + minStep);
   vec2 gridPnt3 = vec2(grid.x + minStep, grid.y);
   vec2 gridPnt4 = vec2(gridPnt3.x, gridPnt2.y);

    float s = rand2(grid);
    float t = rand2(gridPnt3);
    float u = rand2(gridPnt2);
    float v = rand2(gridPnt4);
    
    float x1 = smoothstep(0., 1., fract(vl.x));
    float interpX1 = mix(s, t, x1);
    float interpX2 = mix(u, v, x1);
    
    float y = smoothstep(0., 1., fract(vl.y));
    float interpY = mix(interpX1, interpX2, y);
    
    return interpY;
}

float fractalNoise(vec2 vl) {
    float persistance = 2.0;
    float amplitude = 1.;
    float rez = 0.0;
    vec2 p = vl;
    
    for (float i = 0.0; i < OCTAVES; i++) {
        rez += amplitude * valueNoiseSimple(p);
        amplitude /= persistance;
        p *= persistance;
    }
    return rez;
}

float getTLow(float y0, float v0, float g) {
    
    // First solution is 0, second one is 2 * v0 / g
    return 2. * v0 / g;
}

float physTest3D(vec3 coord, vec3 start, float v0, float g, float r,
                out int numBumps, out float inBump) {
	float Gclr = 0.;
    float lowerPnt = getTLow(start.y, v0, g);
    float time = iGlobalTime / 3.;
    
    int attenuation = int(time / lowerPnt);

    attenuation = int(mod(float(attenuation), 3.));
    
    vec3 center = start;
    float y_offset = 0.;
    
    if (v0 > 0.5) {
        v0 /= float(attenuation + 1);
        lowerPnt = getTLow(start.y, v0, g);
    	time = mod(time, lowerPnt);
        
       	y_offset = v0 * time - (g * time * time) / 2.;
        
   		center.y = start.y + y_offset;
    }
    
    center.y += r;
    
    numBumps = attenuation;
    inBump = time;
    
    return object11(coord, center, r);
}

float sceneSimple(vec3 a) {
    float res = 10.;
    float Floor = -0.98;
    float Step = 20. / NUM_BALLS;
    
    float idxX = floor(a.x / Step);
    float idxZ = floor(a.z / Step);
    a.x = mod(a.x, Step);
    a.z = mod(a.z, Step);
    
    float r = 0.1;
    
    float offsetX = Step / 2. + (rand(vec2(idxX, idxZ))* 2. - 1.) * (Step / 2. - 2. * r);
    float offsetZ = Step / 2. + (rand2(vec2(idxX, idxZ))* 2. - 1.) * (Step / 2. - 2. * r);

    int tmp1;
    float tmp2;
    
    res = physTest3D(a,
                     vec3(offsetX, Floor, offsetZ), 
                     4. + 2. * (rand(vec2(idxX / 10., idxZ/ 10.)) * 2. - 1.),
                     9.8,
                     r,
                     tmp1,
                     tmp2);
    
    
    return min(res, a.y - Floor);
}

float scene(vec3 a, out objectInfo inf) {
    float res = 10.;
    float Floor = -0.98;
    float Step = 20. / NUM_BALLS;
    
    float idxX = floor(a.x / Step);
    float idxZ = floor(a.z / Step);
    a.x = mod(a.x, Step);
    a.z = mod(a.z, Step);
    
    float r = 0.1;
    
    float offsetX = Step / 2. + (rand(vec2(idxX, idxZ))* 2. - 1.) * (Step / 2. - 2. * r);
    float offsetZ = Step / 2. + (rand2(vec2(idxX, idxZ))* 2. - 1.) * (Step / 2. - 2. * r);
                          
    res = physTest3D(a,
                     vec3(offsetX, Floor, offsetZ), 
                     4. + 2. * (rand(vec2(idxX / 10., idxZ/ 10.)) * 2. - 1.),
                     9.8,
                     r,
                     inf.numBumps,
                     inf.inBump);
    
    if (res > (a.y - Floor)) {
        res = a.y - Floor;
        inf.id = vec2(0.);
    } else {
        inf.id.x = idxX + 1.;
        inf.id.y = idxZ + 1.;
    }    
    
    return res;
}

vec3 snormal(vec3 a) {
   vec2 e = vec2(.0001, 0.);
   float tmp1;
   float tmp2;
   float w = sceneSimple(a);

    return normalize(vec3(
       sceneSimple(a+e.xyy) - w,
       sceneSimple(a+e.yxy) - w,
       sceneSimple(a+e.yyx) - w));
}

float trace(vec3 O, vec3 D, out objectInfo inf) {
    
    float L = 0.;
    int steps = 0;
    float d = 0.;
    for (int i = 0; i < MAX_STEPS; ++i) {
        d = scene(O + D*L, inf);
        d = clamp(d, 0., .15);
        L += d;
        
        if (d < THRESHOLD * L) // Adaptive threshold 
            break;
    }
    
    inf.hill = d;
    return L;
}

float occluded(vec3 p, float len, vec3 dir) {
    return max(0., len - sceneSimple(p + len * dir));
}

float occlusion(vec3 p, vec3 normal) {
    vec3 rotZccw = vec3(-normal.y, normal.xz);
    vec3 rotZcw = vec3(normal.y, -normal.x, normal.z);
    
    vec3 rotXccw = vec3(normal.x, normal.z, -normal.y);
    vec3 rotXcw = vec3(normal.x, -normal.z, normal.y);
    
    vec3 rotYccw = vec3(normal.z, normal.y, -normal.x);
    vec3 rotYcw = vec3(-normal.z, normal.y, normal.x);
    
    float rez = 0.;
    float dst = .2;

   	rez+= occluded(p, dst, normal);
    
    rez+= occluded(p, dst, rotXccw);
    rez+= occluded(p, dst, rotXcw);

    rez+= occluded(p, dst, rotYccw);
    rez+= occluded(p, dst, rotYcw);

    rez+= occluded(p, dst, rotZccw);
    rez+= occluded(p, dst, rotZcw);

    return (1. - min(rez, 1.));
}

vec3 enlight(vec3 p, vec3 normal, vec3 eye, vec3 lightPos, objectInfo inf) {
    vec3 dir = lightPos - p;
    vec3 eyeDir = eye - p;
    vec3 I = normalize(dir);
    
    vec3 color;
    float specMat = 0.3;
    
    if (inf.id.x == 0. && inf.id.y == 0.) {
        vec3 txtP = p;
        txtP.z += .75 * inf.timeOffset;
        color = texture2D(iChannel0, txtP.xz *.5 + .5).rgb;
    } else {
        float threshold = 0.15;
        float bumps = float(inf.numBumps) / 3.;
        
        color.r = mod(rand(vec2(inf.id.x / 1000.)) + bumps, 1.);
        color.g = mod(rand(vec2(inf.id.y / 1000.)) + bumps, 1.);
        color.b = mod(rand(vec2((inf.id.x + inf.id.y) / 1000.)) + bumps, 1.);

        specMat = clamp(rand(vec2((inf.id.x + inf.id.y) / 1000.)), 0.01, 1.);
        
        float rnd = rand(inf.id / 1000.);

        vec3 offset = vec3(rand(p.xz) * 2. -1.,
                           rand(p.xy) * 2. -1.,
                           rand2(p.xz)* 2. -1.);

        if (rnd > 0.75) {
            float noise = fractalNoise(p.xy * 50. + rnd * 10.);
            color *= noise;
            normal *= noise;
        } else {
            normal *= clamp(rnd + 0.5, .5, 1.);
        }

        if (inf.inBump < threshold) {
            
            color = mix(vec3(.5, .0, .0),
                        color,
                        inf.inBump / threshold);
        }
    }
    
    vec3 ambient = color;
    vec3 diffuse = max(dot(normal, I), 0.) * color.rgb;

    diffuse = clamp(diffuse, 0., 1.) * 0.5;

    vec3 refl = normalize(-reflect(I, normal));
    float spec = max(dot(refl, normalize(eyeDir)), 0.);
    
    spec = pow(spec, specMat * 60.);
    spec = clamp(spec, 0., 1.);
    
    vec3 Ispec = spec * vec3(1.0, 1.0, 1.0);
    
    return Ispec + diffuse + ambient * occlusion(p, normal);
}

void main(void)
{
    vec2 uv = gl_FragCoord.xy / iResolution.xy;
    vec2 centered_uv = uv * 2. - 1.;
    centered_uv.x *= iResolution.x / iResolution.y;
    
    float timeOffset = iGlobalTime * 2.;
    
    vec3 O = vec3(0., 0.1, 2. - timeOffset);
    vec3 D = normalize(vec3(centered_uv, -3.5)); //fov
    
    objectInfo inf;
    float path = trace(O, D, inf);
    vec3 coord = O + path * D;
    inf.timeOffset = timeOffset;
    
    vec3 skyBlueColor = vec3(0.529411765, 0.807843137, 0.980392157);
    vec3 color = mix(skyBlueColor, vec3(1.), clamp(centered_uv.y, 0., 1.));
    
    if (inf.hill >= 0.15) {
        gl_FragColor = vec4(color , 1.);
    } else {
        vec3 lightPos = vec3(0., 10., -4. - timeOffset);
        vec3 normal = snormal(coord);

        vec3 resColor = enlight(coord, normal, O, lightPos, inf);
        // Calc some fog
    	float fogFactor = exp(-pow(abs(fogDensity * (coord.z - 1. + timeOffset)), 4.0));
    	fogFactor = clamp(fogFactor, 0.0, 1.0);
    	resColor = mix(color, resColor, fogFactor);
        
        gl_FragColor = vec4(resColor,
            				1.0);
    }
}